/**
*    LED_Cube
*    
*    Code for an Arduino Mega to control an LED cube.  Ports 0 to 48 should 
*    switch the columns, with (0,0 to 6) going to ports 0 to 6,
*    (1,0 to 6) going to ports 7 to 13, etc.
*    
*    Ports A0 through A6 control the layers with A0 controlling z = 0 and A6
*    controlling z = 6.  
*
*    Ports A8 to A15 serve as inputs for the pushbutton switches.
*
*    Program written by Lopuz3
*    Spring 2013 
**/

#include <PushButton.h>
#include <CubeView.h>
#include <Routine.h>
#include <AllOn.h>
#include <CornerToCorner.h>
#include <CubeFrame.h>
#include <Fade.h>
#include <Ripple.h>
#include <Wave.h>
#include <PouringRain.h>
#include <BouncingBall.h>
#include <Particle.h>
#include <Firework.h>

#define CUBE_SIZE 7
#define NUM_BUTTONS 8
#define REFRESH_RATE 60

PushButton buttons[NUM_BUTTONS];
CubeView cube = CubeView();
Routine *currentRoutine = new Routine();

boolean lastCompleteFrame[CUBE_SIZE][CUBE_SIZE][CUBE_SIZE];

void setup() 
{
  setUpButtons();
  setUpInterrupts();
}

void setUpInterrupts()
{
  cli();//stop interrupts while we set them up
  //set up an interrupt with timer1
  TCCR1A = 0;
  TCCR1B = 0;
  TCNT1  = 0;
  //make the interrupt ocurr at the correct frequency.  The frequency is REFRESH_RATE*CUBE_SIZE
  OCR1A = (16000000/REFRESH_RATE/1024/CUBE_SIZE -1);
  TCCR1B |= (1 << WGM12);
  // Set to CS10 and CS12 so we have the 1024
  TCCR1B |= (1 << CS12) | (1 << CS10);  
  TIMSK1 |= (1 << OCIE1A);
  sei();//reallow interrupts 
}

void setUpButtons()
{
  for(byte i = 0 ; i < NUM_BUTTONS ; i++)
  {
    buttons[i] = PushButton(i+A8);
  }
}

//Called by timer interrupt
ISR(TIMER1_COMPA_vect)
{
  cube.displayLayer(lastCompleteFrame);
}

void loop() 
{
  currentRoutine->update(getTimeSinceLastFrameInMicros());
  memcpy(&lastCompleteFrame, &currentRoutine->cubeModel, sizeof(boolean)*CUBE_SIZE*CUBE_SIZE*CUBE_SIZE);
  getButtonInput();
}

void getButtonInput()
{
  if(anyButtonWasPressed())
  {
    delete currentRoutine;
    currentRoutine = new AllOn();
  }
  if(buttons[0].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new CornerToCorner();
  }
  if(buttons[1].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new CubeFrame();
  }
  if(buttons[2].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new Fade();
  }
  if (buttons[3].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new Ripple();
  }
  if (buttons[4].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new Wave();
  }   
  if (buttons[5].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new PouringRain();
  }
  if (buttons[6].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new BouncingBall();
  }
  if (buttons[7].wasReleased())
  {
    delete currentRoutine;
    currentRoutine = new Firework();
  } 
}

boolean anyButtonWasPressed()
{
  for(byte i = 0 ; i < NUM_BUTTONS ; i++)
  {
    if(buttons[i].wasPressed())
    {
      return true;
    }
  }
  return false;
}

/*Get the time since the last call of this function in microseconds*/
unsigned long getTimeSinceLastFrameInMicros()
{
  static unsigned long lastTime = 0;
  unsigned long dt = micros()-lastTime;
  lastTime = micros();
  return dt;
}
